<?php
/**
 * Admin Class
 *
 * Handles the admin interface for CrawlConsole Analytics
 */
class CrawlConsole_Analytics_Admin {

    
    public function init() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_filter('plugin_action_links_' . plugin_basename(dirname(__DIR__) . '/crawlconsole-analytics.php'), array($this, 'add_settings_link'));

        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }
    
    public function enqueue_admin_assets($hook) {
        if ('settings_page_crawlconsole-analytics' !== $hook) {
            return;
        }
        
        wp_enqueue_style(
            'crawlconsole-analytics-admin',
            CRAWLCONSOLE_ANALYTICS_URL . 'admin/css/crawlconsole-analytics-admin.css',
            array(),
            CRAWLCONSOLE_ANALYTICS_VERSION
        );
        
        wp_enqueue_script(
            'crawlconsole-analytics-admin',
            CRAWLCONSOLE_ANALYTICS_URL . 'admin/js/crawlconsole-analytics-admin.js',
            array('jquery'),
            CRAWLCONSOLE_ANALYTICS_VERSION,
            true
        );
        
        wp_localize_script(
            'crawlconsole-analytics-admin',
            'crawlconsole_analytics_vars',
            array(
                'disable_confirmation' => __('Are you sure you want to disable AI Analytics tracking? No data will be sent to Crawlconsole while disabled.', 'crawlconsole-analytics'),
                'copy_text' => __('Copy', 'crawlconsole-analytics'),
                'copied_text' => __('Copied!', 'crawlconsole-analytics'),
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('crawlconsole_analytics_nonce')
            )
        );
    }

    public function add_admin_menu() {
        add_options_page(
            'CrawlConsole Analytics', 
            'CrawlConsole Analytics', 
            'manage_options', 
            'crawlconsole-analytics', 
            array($this, 'admin_page')
        );
    }

    public function register_settings() {
        register_setting(
            'crawlconsole_analytics_options',
            'crawlconsole_analytics_api_key',
            array(
                'type' => 'string',
                'description' => 'CrawlConsole API Key for AI Analytics tracking',
                'sanitize_callback' => function($input) {
                    $sanitized = sanitize_text_field(wp_unslash($input));
                    
                    if (empty($sanitized)) {
                        add_settings_error(
                            'crawlconsole_analytics_api_key',
                            'empty_api_key',
                            __('API key cannot be empty.', 'crawlconsole-analytics')
                        );
                        return get_option('crawlconsole_analytics_api_key'); // Keep the old value
                    }
                    
                    return $sanitized;
                },
                'show_in_rest' => false,
                'default' => ''
            )
        );
        register_setting(
            'crawlconsole_analytics_options',
            'crawlconsole_analytics_project_key',
            array(
                'type' => 'string',
                'description' => 'Crawlconsole Project Key for AI Analytics tracking',
                'sanitize_callback' => function($input) {
                    $sanitized = sanitize_text_field(wp_unslash($input));
                    if (empty($sanitized)) {
                        add_settings_error(
                            'crawlconsole_analytics_project_key',
                            'empty_project_key',
                            __('Project Key cannot be empty.', 'crawlconsole-analytics')
                        );
                        return get_option('crawlconsole_analytics_project_key');
                    }
                    return $sanitized;
                },
                'show_in_rest' => false,
                'default' => ''
            )
        );
    }


    public function add_settings_link($links) {
        $settings_link = '<a href="options-general.php?page=crawlconsole-analytics">' . __('Settings', 'crawlconsole-analytics') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Render admin page
     */
    public function admin_page() {
        $api_key = get_option('crawlconsole_analytics_api_key', '');
        $project_key = get_option('crawlconsole_analytics_project_key', '');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="promptalytics-admin-card">
                <p style="margin-bottom: 1.5rem; font-size: 1rem; color: #374151;">
                    <?php esc_html_e('Get your CrawlConsole API key from your account ', 'crawlconsole-analytics'); ?> 
                    <a href="https://crawlconsole.com/settings/" target="_blank" rel="noopener noreferrer" style="color: #2563eb; text-decoration: underline;">
                        here
                    </a>
                </p>
                <form method="post" action="options.php" id="crawlconsole_analytics_form">
                    <?php settings_fields('crawlconsole_analytics_options'); ?>
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="crawlconsole_analytics_api_key"><?php esc_html_e('API Key', 'crawlconsole-analytics'); ?></label>
                            </th>
                            <td>
                                <div>
                                    <input type="password" 
                                           id="crawlconsole_analytics_api_key" 
                                           name="crawlconsole_analytics_api_key" 
                                           value="<?php echo esc_attr($api_key); ?>" 
                                           class="regular-text" />
                                    <i class="fa fa-eye" id="toggle-password" style="cursor: pointer;"></i>
                                </div>
                                <p class="description-text"><?php esc_html_e('Enter the API key provided by CrawlConsole.', 'crawlconsole-analytics'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="crawlconsole_analytics_project_key"><?php esc_html_e('Project Key', 'crawlconsole-analytics'); ?></label>
                            </th>
                            <td>
                                <div>
                                    <input type="text" 
                                           id="crawlconsole_analytics_project_key" 
                                           name="crawlconsole_analytics_project_key" 
                                           value="<?php echo esc_attr($project_key); ?>" 
                                           class="regular-text" />
                                </div>
                                <p class="description-text"><?php esc_html_e('Enter the Project Key provided by CrawlConsole.', 'crawlconsole-analytics'); ?></p>
                            </td>
                        </tr>
                    </table>
                    <?php submit_button(esc_html__('Save Settings', 'crawlconsole-analytics'), 'button-class'); ?>
                </form>
            </div>
        </div>
        <?php
    }
}