<?php

class CrawlConsole_Analytics_Request_Middleware {

    private $analytics_data = array();
    private $status_code = null;

    public function init() {
        add_action('template_redirect', array($this, 'intercept_request'), 1);
    }

    public function intercept_request() {
        if (!get_option('crawlconsole_analytics_enabled', true)) {
            return;
        }

        $this->__construct();
    }

    public function __construct() {
        add_filter('status_header', array($this, 'capture_status_code'), 10, 2);
        $this->analytics_data = $this->gather_request_data();
        $this->send_analytics_async();
    }

    private function send_analytics_async() {
        add_action('shutdown', array($this, 'send_analytics_data'));
    }

    public function send_analytics_data() {
        if (is_admin()) {
            return;
        }

        $analytics_data = $this->analytics_data;
        $api_key = get_option('crawlconsole_analytics_api_key', '');

        $api = new CrawlConsole_Analytics_API();

        $payload = array(
            'project_key'  => $analytics_data['project_key'] ?? '',
            'user_agent'   => $analytics_data['user_agent'] ?? '',
            'path'         => $analytics_data['path'] ?? '',
            'full_path'    => $analytics_data['full_path'] ?? '',
            'headers'      => (object)[],
            'client_ip'    => $analytics_data['client_ip'] ?? '',
            'client_ip_raw'=> $analytics_data['client_ip_raw'] ?? '',
            'status_code'  => $this->status_code,
        );
        $api->send_analytics($payload, $api_key);
    }

    private function gather_request_data() {
        $all_headers = $this->get_all_headers();
        $user_agent = $all_headers['User-Agent'] ?? ($_SERVER['HTTP_USER_AGENT'] ?? '');
        // Robust client IP detection
        $client_ip = $this->get_real_client_ip();
        $client_ip_raw = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? '';
        return array(
            'user_agent' => $user_agent,
            'headers'    => $all_headers,
            'path'       => $_SERVER['REQUEST_URI'] ?? '',
            'full_path'  => (isset($_SERVER['HTTP_HOST']) ? ($_SERVER['HTTP_HOST'] . ($_SERVER['REQUEST_URI'] ?? '')) : ''),
            'client_ip'  => $client_ip,
            'client_ip_raw' => $client_ip_raw,
            'project_key'=> get_option('crawlconsole_analytics_project_key', ''),
        );
    }

    // Returns the best guess for the real client IP, considering trusted proxy headers
    private function get_real_client_ip() {
        // Check for X-Forwarded-For header (may be a comma-separated list)
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $forwarded_ips = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
            // The first IP in the list is the original client (if trusted proxy)
            $ip = trim($forwarded_ips[0]);
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
        // Check for X-Real-IP header
        if (!empty($_SERVER['HTTP_X_REAL_IP']) && filter_var($_SERVER['HTTP_X_REAL_IP'], FILTER_VALIDATE_IP)) {
            return $_SERVER['HTTP_X_REAL_IP'];
        }
        // Fallback to REMOTE_ADDR
        return $_SERVER['REMOTE_ADDR'] ?? '';
    }

    private function get_all_headers() {
        if (function_exists('getallheaders')) {
            return getallheaders();
        }

        $headers = array();
        foreach ($_SERVER as $name => $value) {
            if (substr($name, 0, 5) === 'HTTP_') {
                $header_name = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
                $headers[$header_name] = sanitize_text_field(wp_unslash($value));
            } elseif ($name === 'CONTENT_TYPE' || $name === 'CONTENT_LENGTH') {
                $header_name = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', $name))));
                $headers[$header_name] = sanitize_text_field(wp_unslash($value));
            }
        }
        return $headers;
    }

    private function get_client_ip() {
        $ip_keys = array(
            'HTTP_CF_CONNECTING_IP',
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR',
        );

        foreach ($ip_keys as $key) {
            if (isset($_SERVER[$key]) && filter_var(wp_unslash($_SERVER[$key]), FILTER_VALIDATE_IP)) {
                return sanitize_text_field(wp_unslash($_SERVER[$key]));
            }
        }

        return '';
    }

    public function capture_status_code($status_header, $code) {
        $this->status_code = $code;
        return $status_header;
    }

    private function get_country_code() {
        $headers = $this->get_all_headers();
        if (!empty($headers['CF-IPCountry'])) {
            return $headers['CF-IPCountry'];
        }

        $country_headers = array(
            'HTTP_X_COUNTRY_CODE',
            'HTTP_GEOIP_COUNTRY_CODE',
            'HTTP_X_COUNTRY',
        );
        foreach ($country_headers as $header) {
            if (isset($_SERVER[$header])) {
                return sanitize_text_field(wp_unslash($_SERVER[$header]));
            }
        }

        if (function_exists('geoplugin_countryCode')) {
            $country_code = geoplugin_countryCode();
            if (!empty($country_code)) {
                return $country_code;
            }
        }

        return '';
    }
}
